#!/bin/bash
#
# This script installs and configures Twistlock
#

# Silence tput errors when script is run by process not attached to a terminal
tput_silent() {
	tput "$@" 2>/dev/null
}

if [ "$(id -u)" != "0" ]; then
	echo "$(tput_silent setaf 1)This script must be run as root $(tput_silent sgr0)"
	exit 1
fi

# Global initialization
# Working folder is the folder where Twistlock scripts and images reside
working_folder=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)
config_file=${working_folder}"/twistlock.cfg"
syslog_mount="/dev/log"
if [ -e "${config_file}" ]; then
	source ${config_file}
else
	echo "Configuration file ${config_file} must be present"
	exit
fi

console_bundle=twistlock_console.tar.gz
defender_bundle_linux_docker=twistlock_defender.tar.gz
defender_bundle_linux_server=twistlock_defender_server.tar.gz

console_service=twistlock-console.service
defender_service=twistlock-defender.service
server_defender_service=twistlock-defender-server.service
server_defender_upstart=defender-server.conf

onebox_component=onebox
console_component=console
defender_component_docker=defender
defender_component_podman=defender-podman
defender_component_linux_server=defender-server
defender_component_tas=defender-tas

# docker's security opt operator, to be used in '--security-opt label${security_opt_operator}role:ROLE'.
# Different docker versions has different values ('=' vs ':')
security_opt_operator="="

# Global flags
# Debug log level
debug_log_level="false"
# Location of Twistlock install logs
install_log="twistlock-install.log"
console_container_name=twistlock_console

# Container runtime cli command to use
docker="docker"

# Additional environment variables for Defender/Console containerized deployments
additional_env=""

static_data_folder="/prisma-static-data"

is_docker() {
	# Check if command starts with "docker"
	[[ "${docker}" == docker* ]]
}

is_podman() {
	# Check if command starts with "podman"
	[[ "${docker}" == podman* ]]
}

# Silence tput errors when script is run by process not attached to a terminal
tput_silent() {
	tput "$@" 2>/dev/null
}

print_debug() {
	debug=$1
	if [[ ${debug_log_level} == "true" ]]; then
		echo "$(tput_silent setaf 7)${debug}.$(tput_silent sgr0)"
	fi
	echo ${debug} >>${install_log}
}

# checks whether the given path is a docker socket or not
is_valid_docker_socket() {
	socketPath=$1
	if [ ! -S ${socketPath} ]; then
		return 1
	fi
	docker -H unix://${socketPath} info >>/dev/null 2>&1
}

# Try to get Docker version using format flag. For older versions it may fail because format flag doesnt exist (self parse it)
set_docker_version() {
	docker_version=$(${docker} version --format '{{.Server.Version}}' 2>/dev/null)
	if [[ $? != 0 ]]; then
		docker_version=$(${docker} version | grep -i -A1 '^server' | grep -i 'version:' | awk '{print $NF; exit}' | tr -d '[:alpha:]-,')
	fi
}

container_runtime_socket_folder=$(dirname ${DOCKER_SOCKET})
new_docker_socket="${container_runtime_socket_folder}/docker.tw.sock"

# Sets the docker command to variable 'docker' after stopping the defender. Regardless of local socket mode, the socket should be at the original path
set_docker_command_on_defender_stop() {
	if ! is_docker; then
		return
	fi

	is_valid_docker_socket ${new_docker_socket}
	if [[ $? == 0 ]]; then
		exit_now "Failed to set valid docker socket, docker socket shouldn't be at ${new_docker_socket}"
	fi

	is_valid_docker_socket ${DOCKER_SOCKET}
	exit_on_failure $? "Failed to find valid docker socket"

	if [[ $? == 0 ]]; then
		docker="docker -H unix://${DOCKER_SOCKET}"
		print_debug "setting docker command after Defender stop to ${docker}"
		return
	fi
}

# Sets the docker command to variable 'docker' by testing naively the possible docker sockets
set_docker_command() {
	local component=${1}
	if [ "${component}" == "${defender_component_podman}" ]; then
		docker="podman"
		print_debug "setting initial docker command to ${docker}"
		return
	fi

	is_valid_docker_socket ${new_docker_socket}
	if [[ $? == 0 ]]; then
		docker="docker -H unix://${new_docker_socket}"
		print_debug "setting initial docker command to ${docker}"
		return
	fi

	is_valid_docker_socket ${DOCKER_SOCKET}
	if [[ $? == 0 ]]; then
		docker="docker -H unix://${DOCKER_SOCKET}"
		print_debug "setting initial docker command to ${docker}"
		return
	fi

	# checks whether podman is running
	podman -v >>/dev/null 2>&1
	if [[ $? == 0 ]]; then
		docker="podman"
		print_debug "setting initial docker command to ${docker}"
		return
	fi

	exit_now "Failed to find valid docker socket or podman"
}

# Default os distribution
host_os_distribution=${HOST_OS_DISTRIBUTION:-$(grep DISTRIB_ID /etc/*-release | awk -F '=' '{print $2}')}
# This comment contains a dummy closing apostrophe to help editors(vi, goland) with parsing the script '

# SELinux mode specifies the SELinux enforcement mode
selinux_enabled=

defender_image=""
console_image=""

print_info() {
	info=$1
	echo "$(tput_silent setaf 2)${info}.$(tput_silent sgr0)"
	echo ${info} >>${install_log}
}

print_error() {
	error=$1
	echo "$(tput_silent setaf 1)${error}.$(tput_silent sgr0)"
	echo ${error} >>${install_log}
}

print_warning() {
	warning=$1
	echo "$(tput_silent setaf 3)${warning}.$(tput_silent sgr0)"
	echo ${warning} >>${install_log}
}

approve_disclaimer() {
	echo "
Dear Customer,

Please review the license terms included in the provided twistlock-license.pdf file.

If you agree to the terms listed in the file please type 'agree' to continue, otherwise please press Ctrl-C to
abort the installation.
"
	read agree

	if [ "${agree}" != "agree" ]; then
		echo "The terms were denied. Exiting..."
		exit 1
	else
		echo "Thanks for agreeing! Continuing the installation..."
	fi
}

remove_container() {
	container_name=$1
	remove_image=$2

	# Find container lines in docker ps output
	container=$(${docker} ps -a --format "{{ .Names }}" | grep ${container_name})
	print_debug "Removing container ${container}"

	# Check if container is running
	running=$(${docker} inspect --format="{{ .State.Running }}" ${container} 2>/dev/null)

	if [ $? -eq 1 ]; then
		print_debug "Container ${container} does not exist"
		return
	fi

	if [ "${running}" == "true" ]; then
		print_debug "Container ${container} is running, stopping it"
		${docker} stop ${container} >>${install_log} 2>&1
	fi

	# If the removed container was defender in local socket mode, then he restored the docker socket when stopping to $DOCKER_SOCKET
	if [[ ${container_name} == "twistlock_defender" ]]; then
		set_docker_command_on_defender_stop
	fi

	if [[ "${remove_image}" == "true" ]]; then
		image=$(${docker} inspect --format="{{ .Image }}" ${container} 2>/dev/null)
	fi

	${docker} rm -f -v ${container} >>${install_log} 2>&1

	if [[ "${remove_image}" == "true" ]]; then
		print_debug "Removing container ${container} image ${image}"
		${docker} rmi ${image}
	fi
}

# validate a container is up and not restarting with a retry mechanism with up to three attempts
validate_container_up() {
	container=$1
	local retries=$2
	print_debug "Verifying container ${container} is up"

	# Check if container is running
	running=$(${docker} inspect --format="{{ .State.Running }}" ${container} 2>/dev/null)
	if [ $? -eq 1 ]; then
		exit_now "Container ${container} does not exist. Deployment failed"
	fi

	local n=0
	until [[ ${n} -ge ${retries} ]]; do
		running=$(${docker} inspect --format="{{ .State.Running }}" ${container} 2>/dev/null)
		if [ "${running}" == "true" ]; then
			break
		fi
		error_msg="Container ${container} is not running; ${running}"
		print_warning "${error_msg}. Retrying..."
		sleep 1

		n=$((${n} + 1))
	done
	if [ ${n} -ge ${retries} ]; then
		exit_now "${error_msg}. Deployment failed"
	fi

	restart_count=$(${docker} inspect --format="{{ .RestartCount }}" ${container} 2>/dev/null)
	if [ "${restart_count}" != "0" ]; then
		exit_now "Container ${container} has restarted multiple times (${restart_count}); Deployment failed"
	fi
}

# Replace placeholder in services configuration file
fill_service_placeholders() {
	service_path=${1}
	sed -e "s#__DATA_FOLDER__#${DATA_FOLDER}#g" \
		-e "s#__INSTALL_FOLDER__#${install_folder}#g" \
		-i "${service_path}"
}

install_console() {
	console_address=$1
	drop_db=$2

	set_data_folder_permissions

	print_info "Initializing Twistlock environment"
	# Set feature compatibility if previous console is running
	# As of Barracuda, console is running with mongo version 4.0 (require feature compatibility) and automatically upgrades to latest compatible version
	# https://docs.mongodb.com/manual/release-notes/4.0-upgrade-standalone/
	if [ -e "/var/lib/twistlock/certificates/service-parameter" ]; then
		${docker} exec -ti ${console_container_name} mongo --ssl --sslAllowInvalidHostnames --sslCAFile /var/lib/twistlock/certificates/ca.pem --sslPEMKeyFile /var/lib/twistlock/certificates/client.pem --sslPEMKeyPassword $(cat /var/lib/twistlock/certificates/service-parameter) ${port_data} \
			--eval 'let compVersion = db.adminCommand( { getParameter: 1, featureCompatibilityVersion: 1 } ).featureCompatibilityVersion.version; if (compVersion == "3.4") { db.adminCommand( { setFeatureCompatibilityVersion: "3.6" } );}' >>${install_log} 2>&1
	fi

	remove_container ${console_container_name}

	additional_ports=""
	additional_parameters=""

	print_debug "Current version ${docker_version}"
	if [[ ${READ_ONLY_FS} != "false" ]]; then
		# https://blog.docker.com/2015/02/docker-1-5-ipv6-support-read-only-containers-stats-named-dockerfiles-and-more/
		print_debug "Setting Console container to readonly"
		additional_parameters+=" --read-only=true "
	else
		print_warning "Console container file-system is not read-only"
	fi

	# Mount external data to /var/lib/twistlock/
	db_folder="${DATA_FOLDER}/db"
	if [[ ${drop_db} == "true" ]]; then
		print_debug "Dropping Twistlock database"
		rm -rf ${db_folder}
		exit_on_failure $? "Failed to delete database folder ${db_folder}"
	fi

	print_debug "Installing database under ${db_folder}"
	mkdir -p ${db_folder}
	exit_on_failure $? "Failed to create database folder ${db_folder}"
	print_info "Installing Twistlock Console (${CONSOLE_CN})"

	local_ips=$(get_local_ip)
	san=${local_ips}
	print_debug "SAN: ${san}"
	print_debug "Console CN: ${CONSOLE_CN}"

	if [[ ${DATA_RECOVERY_ENABLED} == "true" ]]; then
		print_debug "Data recovery is enabled"
		set_data_recovery_folder_permissions
		# Mount data recovery volume and database folder
		additional_parameters+=" -v ${DATA_RECOVERY_VOLUME}:/var/lib/twistlock-backup "

	fi

	if [[ -n ${selinux_enabled} ]]; then
		additional_parameters+=" --security-opt label${security_opt_operator}${SELINUX_LABEL} "
	fi

	# Set pids limit to 1000 so attackers won't be able to launch a fork bomb with a single command inside the container
	if [[ $(version_higher_equal ${docker_version} 1.12) == 1 ]]; then
		# Mongo starts a thread per connection, enable multiple simulations connections to support high load
		additional_parameters+=" --pids-limit 1000 "
	fi

	docker_action=""
	if [[ ${SYSTEMD_ENABLED} == "true" ]]; then
		print_debug "Systemd enabled. creating container"
		docker_action="create"
	else
		docker_action="run"
		additional_parameters+=" -d --restart=always "
	fi

	if [[ ! -z ${MANAGEMENT_PORT_HTTP} ]]; then
		print_debug "http is enabled ${MANAGEMENT_PORT_HTTP}"
		additional_ports+=" -p ${MANAGEMENT_PORT_HTTP}:${MANAGEMENT_PORT_HTTP} "
	else
		print_debug "http is disabled"
	fi

	# Set console disable limits mode based on cgroup limits flag
	DISABLE_LIMITS=${DISABLE_CONSOLE_CGROUP_LIMITS}

	cgroup_limit=${CGROUP_LIMIT}
	if [[ -z ${cgroup_limit} ]]; then
		cgroup_limit="4096m"
	fi

	console_cgroup_limits="--cpu-shares 900 --memory ${cgroup_limit}"

	if [[ ${DISABLE_CONSOLE_CGROUP_LIMITS} == "true" ]]; then
		console_cgroup_limits=""
	fi

	# Create folders that are shared between the defender and console (required to ensure defender does not modify
	# the folder permissions when it starts)
	mkdir -p ${DATA_FOLDER}/certificates && mkdir -p ${DATA_FOLDER}/log
	chown -R ${twistlock_user} ${DATA_FOLDER} && chmod 0700 -R ${DATA_FOLDER}

	if [[ -e ${syslog_mount} ]]; then
		additional_parameters+=" -v ${syslog_mount}:${syslog_mount} "
	fi

	if [[ ${CLUSTERED_DB_ENABLED} == "true" ]]; then
		print_debug "Running in clustered DB mode"
		additional_ports+=" -p 27017:27017 "
		additional_env+=" -e CLUSTERED_DB_ENABLED=${CLUSTERED_DB_ENABLED} "
	fi

	if [[ -n "${AGENTLESS_MAX_SCAN_WORKERS}" ]]; then
		additional_env+=" -e AGENTLESS_MAX_SCAN_WORKERS=${AGENTLESS_MAX_SCAN_WORKERS} "
	fi

	if [[ -n "${REGISTRY_IMAGES_CAP}" ]]; then
		additional_env+=" -e REGISTRY_IMAGES_CAP=${REGISTRY_IMAGES_CAP} "
	fi

	if [[ -n "${AGENTLESS_VERIFY_REGION_CONNECTIVITY}" ]]; then
		additional_env+=" -e AGENTLESS_VERIFY_REGION_CONNECTIVITY=${AGENTLESS_VERIFY_REGION_CONNECTIVITY} "
	fi

	if [[ -n "${REGISTRY_SCANNERS}" ]]; then
		additional_env+=" -e REGISTRY_SCANNERS=${REGISTRY_SCANNERS} "
	fi

	if [[ -n "${REGISTRY_SCAN_DEFENDER_CONCURRENCY}" ]]; then
		additional_env+=" -e REGISTRY_SCAN_DEFENDER_CONCURRENCY=${REGISTRY_SCAN_DEFENDER_CONCURRENCY} "
	fi

	if [[ -n "${REGISTRY_CHANNEL_SIZE}" ]]; then
		additional_env+=" -e REGISTRY_CHANNEL_SIZE=${REGISTRY_CHANNEL_SIZE} "
	fi

	if [[ -n "${BUILDAH_DETECTION_ENABLED}" ]]; then
		additional_env+=" -e BUILDAH_DETECTION_ENABLED=${BUILDAH_DETECTION_ENABLED} "
	fi

	if [[ -n "${CVE_EXTENDED_CONDITIONS}" ]]; then
		additional_env+=" -e CVE_EXTENDED_CONDITIONS=${CVE_EXTENDED_CONDITIONS} "
	fi

	if [[ -n "${POOLED_HASH_GENERATOR_ENABLED}" ]]; then
		additional_env+=" -e POOLED_HASH_GENERATOR_ENABLED=${POOLED_HASH_GENERATOR_ENABLED} "
	fi

	if [[ -n "${SYMBOL_EXTRACTION_ENABLED}" ]]; then
		additional_env+=" -e SYMBOL_EXTRACTION_ENABLED=${SYMBOL_EXTRACTION_ENABLED} "
	fi

	if [[ -n "${CORE_DELETE_STALE_ASSETS_ENABLED}" ]]; then
		additional_env+=" -e CORE_DELETE_STALE_ASSETS_ENABLED=${CORE_DELETE_STALE_ASSETS_ENABLED} "
	fi

	if [[ -n "${CORE_DELETE_STALE_IMAGES_ENABLED}" ]]; then
		additional_env+=" -e CORE_DELETE_STALE_IMAGES_ENABLED=${CORE_DELETE_STALE_IMAGES_ENABLED} "
	fi

	if [[ -n "${CORE_PROJECTS_UPDATE_OUTPUT_CHANNEL_SIZE}" ]]; then
		additional_env+=" -e CORE_PROJECTS_UPDATE_OUTPUT_CHANNEL_SIZE=${CORE_PROJECTS_UPDATE_OUTPUT_CHANNEL_SIZE} "
	fi

	if [[ -n "${CORE_PROJECTS_UPDATE_WORKERS}" ]]; then
		additional_env+=" -e CORE_PROJECTS_UPDATE_WORKERS=${CORE_PROJECTS_UPDATE_WORKERS} "
	fi

	if [[ -n "${CORE_SSL_CLIENT_CERT_HEADER_NAME}" ]]; then
		additional_env+=" -e CORE_SSL_CLIENT_CERT_HEADER_NAME=${CORE_SSL_CLIENT_CERT_HEADER_NAME} "
	fi

	if [[ -n "${CORE_SSL_CLIENT_CERT_ENCODING}" ]]; then
		additional_env+=" -e CORE_SSL_CLIENT_CERT_ENCODING=${CORE_SSL_CLIENT_CERT_ENCODING} "
	fi

	if [[ -n "${AGENTLESS_USE_CUSTOM_AMI}" ]]; then
		additional_env+=" -e AGENTLESS_USE_CUSTOM_AMI=${AGENTLESS_USE_CUSTOM_AMI} "
	fi

	if [[ -n "${AGENTLESS_CUSTOM_AMI_PER_REGION}" ]]; then
		additional_env+=" -e AGENTLESS_CUSTOM_AMI_PER_REGION=${AGENTLESS_CUSTOM_AMI_PER_REGION} "
	fi

	# All console containers resides on the same network
	${docker} ${docker_action} \
		$(get_userns_flag) \
		--name ${console_container_name} \
		--user ${twistlock_user} \
		-p ${MANAGEMENT_PORT_HTTPS}:${MANAGEMENT_PORT_HTTPS} \
		-p ${COMMUNICATION_PORT}:${COMMUNICATION_PORT} \
		${additional_ports} \
		-e CONSOLE_CN=${CONSOLE_CN} \
		-e CONSOLE_SAN="${san}" \
		-e LOG_PROD=true \
		-e DATA_RECOVERY_ENABLED=${DATA_RECOVERY_ENABLED} \
		-e COMMUNICATION_PORT=${COMMUNICATION_PORT} \
		-e MANAGEMENT_PORT_HTTPS=${MANAGEMENT_PORT_HTTPS} \
		-e MANAGEMENT_PORT_HTTP=${MANAGEMENT_PORT_HTTP} \
		-e DISABLE_LIMITS=${DISABLE_LIMITS} \
		-e FIPS_ENABLED=${FIPS_ENABLED} \
		-v ${DATA_FOLDER}:/var/lib/twistlock \
		${additional_env} \
		${additional_parameters} \
		${console_cgroup_limits} \
		${console_image} >>${install_log} 2>&1

	exit_on_failure "$?" "Failed to run Twistlock Console"

	local retries=1
	if [[ ${SYSTEMD_ENABLED} == "true" ]]; then
		print_debug "Starting Twistlock Console using systemd"
		install_systemd_service ${console_service}
		retries=3
	fi
	validate_container_up "${console_container_name}" ${retries}
	print_info "Twistlock Console installed successfully"
}

# Copy twistlock scripts to data folder
copy_scripts() {
	local scripts_folder="${DATA_FOLDER}/scripts"
	print_debug "Copying Twistlock configuration and scripts to ${scripts_folder}"
	mkdir -p ${scripts_folder}
	if [ -e "${scripts_folder}/twistlock.cfg" ]; then
		print_info "Previous Twistlock configuration detected, preserving configuration under: ${scripts_folder}/twistlock.cfg.old"
		mv "${scripts_folder}/twistlock.cfg" "${scripts_folder}/twistlock.cfg.old"
	fi
	cp ${working_folder}/twistlock.* ${scripts_folder}
	# Make twistlock script executable
	chmod a+x ${scripts_folder}/twistlock.sh
	# Ensure that cfg file is still readable by console (if defender is installed on the same machine as console, and console run as user)
	chown ${twistlock_user} ${scripts_folder}/twistlock.cfg && chmod u+r ${scripts_folder}/twistlock.cfg
}

# Get all local ips of the current host
get_local_ip() {
	ip=""
	# In some distributions, grep is not compiled with -P support.
	# grep: support for the -P option is not compiled into this --disable-perl-regexp binary
	# For those cases, use pcregrep
	if command_exists pcregrep; then
		ip=$(ip -f inet addr show | pcregrep -o 'inet \K[\d.]+')
	else
		ip=$(ip -f inet addr show | grep -Po 'inet \K[\d.]+')
	fi
	ip_result="IP:"
	ip_result+=$(echo ${ip} | sed 's/ /,IP:/g')
	echo ${ip_result}
}

# Retrun true if current host is a init system (upstart or sysvinit)
# REMARK: Current support is only for upstart
is_init_system() {
	# This command returns either systemd or init and appears reliable across Linux distributions and releases
	# Note that PID 1 is the init/systemd process
	[[ $(ps --no-headers -o comm 1) == "init" ]]
}

# Uninstall all Twistlock containers
uninstall_twistlock() {
	print_info "Uninstalling Twistlock"
	if [[ ${SYSTEMD_ENABLED} == "true" ]]; then
		uninstall_systemd_service ${defender_service}
		uninstall_systemd_service ${console_service}
	fi

	if is_init_system; then
		uninstall_upstart_defender_server
	fi

	remove_container "twistlock_defender" "false"
	remove_container "twistlock_console" "false"

	# Find all images that contains the term twistlock/private
	# Take unique values from IMAGE ID column (column 3 in `docker images`)
	twistlock_images=$(${docker} images | grep twistlock/private | awk '{print $3}' | sort | uniq)

	for image in ${twistlock_images}; do
		print_info "Removing image ${image}"
		${docker} rmi -f ${image}
	done

	print_info "Removing Twistlock data folder: ${DATA_FOLDER}"
	rm -rf ${DATA_FOLDER}
}

get_systemd_unit_path() {
	if [ -d "/lib/systemd/" ]; then
		# Ubuntu, CentOS, Red Hat and Amazon Linux use this path
		echo "/lib/systemd/system"
	elif [ -d "/usr/lib/systemd/" ]; then
		# SUSE Linux uses this path
		echo "/usr/lib/systemd/system"
	fi

}

install_systemd_service() {
	local service=$1
	local unitpath=$(get_systemd_unit_path)
	print_info "Installing systemd service ${service}"
	cp "${DATA_FOLDER}/scripts/${service}" "${unitpath}" >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to copy systemd service unit file"
	chmod 0644 "${unitpath}/${service}" >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to set file permissions for ${service}"
	systemctl daemon-reload >>${install_log} 2>&1
	if [[ $? != 0 ]]; then
		# systemctl daemon-reload connectivity might fail for sporadic reasons
		# see here: https://github.com/systemd/systemd/issues/3353
		sleep 1
		systemctl daemon-reload >>${install_log} 2>&1
		exit_on_failure "$?" "Failed to reload systemctl daemon"
	fi
	systemctl enable ${service} >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to enable ${service}"
	systemctl restart ${service} >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to start ${service}"
	systemctl is-active ${service} >>${install_log} 2>&1
	exit_on_failure "$?" "Service ${service} is not running. Deployment failed"
}

uninstall_systemd_service() {
	local service=$1
	local unitpath=$(get_systemd_unit_path)

	if [[ ! -e "${unitpath}/${service}" ]]; then
		print_info "Service ${service} does not exist"
		return
	fi

	print_info "Uninstalling systemd service ${service}"
	systemctl stop ${service} >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to stop ${service}"
	systemctl disable ${service} >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to disable ${service}"
	rm "${unitpath}/${service}" >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to remove systemd service unit file"
	systemctl daemon-reload >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to reload systemctl daemon"
}

# Return true if upstart defender server service is running
is_upstart_service_running() {
	# Parse the following line to extract service status:
	# defender-server start/running, process 25585
	status=$(initctl status ${defender_component_linux_server} | cut -f2 -d "/" | cut -f1 -d ",")
	[ "${status}" = "running" ]
}

install_upstart_defender_server() {
	local service=${defender_component_linux_server}
	print_info "Installing upstart service ${service}"
	local initpath="/etc/init/"

	cp "${DATA_FOLDER}/scripts/${server_defender_upstart}" "${initpath}" >>${install_log} 2>&1
	exit_on_failure "$?" "Failed to copy upstart service script"

	if is_upstart_service_running; then
		# Restart service is done in case of upgrade
		# This will terminate execution and start the new updated service
		initctl restart "${service}" >>${install_log} 2>&1
		exit_on_failure "$?" "Failed to restart ${service}"
	else
		# Start the service
		initctl start "${service}" >>${install_log} 2>&1
		exit_on_failure "$?" "Failed to start ${service}"
	fi

	# Check that defender server service is running
	for i in {1..5}; do
		# Wait for process to start
		sleep 1
		if is_upstart_service_running; then
			return
		fi
	done

	exit_on_failure 1 "Service ${service} is not running. Deployment failed"
}

uninstall_upstart_defender_server() {
	local service=${defender_component_linux_server}
	local initpath="/etc/init"

	if [[ ! -e "${initpath}/${server_defender_upstart}" ]]; then
		print_info "Service ${service} does not exist"
		return
	fi

	rm -f "${initpath}/${server_defender_upstart}" >>$install_log 2>&1
	exit_on_failure "$?" "Failed to remove upstart service script"

	print_info "Uninstalling upstart service ${service}"
	if is_upstart_service_running; then
		# Following command will stop the service but will return an error since the service file was already removed
		# Note: this is the last step since it terminates the service (which is the process running this script)
		initctl stop "${service}" >>${install_log} 2>&1
	fi
}

install_defender() {
	local console_address=${1}
	local registry_enabled=${2}
	local install_bundle=${3}
	local component=${4}

	if command_exists systemctl; then
		systemctl is-enabled ${server_defender_service} >/dev/null 2>&1
		if [[ $? == 0 ]]; then
			exit_now "Defender for Linux Server is already installed. Please decommission in order to continue"
		fi
	fi
	if is_init_system; then
		initctl status ${defender_component_linux_server} >/dev/null 2>&1
		if [[ $? == 0 ]]; then
			exit_now "Defender for Linux Server is already installed. Please decommission in order to continue"
		fi
	fi
	copy_local_certificates

	set_data_folder_permissions

	print_info "Installing Defender"
	print_debug "Defender CN        : ${DEFENDER_CN}"
	print_debug "Console  address   : ${console_address}"
	print_debug "Registry enabled   : ${registry_enabled}"

	# Remove previous installations
	if [[ ${SKIP_DELETE} == "true" ]]; then
		# Avoid deleting container images to enable running two defender instances simultaneously
		print_warning "Skipping deletion of old containers"
	else
		remove_container "twistlock_defender"
	fi

	if [ -z "${console_address}" ]; then
		console_address="${CONSOLE_CN}"
		print_debug "Console address not provided, assuming onebox mode (IP: ${console_address})"
	fi

	WS_ADDRESS=wss://${console_address}:${COMMUNICATION_PORT}

	# Additional mount points required for performing host CIS scan
	# Explicitly add folder only if exists to avoid failures due to read only host filesystem
	additional_mounts=""

	additional_mounts+=" -v /etc/passwd:/etc/passwd:ro "

	additional_mounts+=" -v ${container_runtime_socket_folder}:${container_runtime_socket_folder} "

	# Ensure runc proxy socket folder is mounted
	# Needed for deployments that use custom mount folder for docker socket
	if [ "${container_runtime_socket_folder}" != "/run" ]; then
		additional_mounts+=" -v /run:/run "
	fi

	# Ensure systemd socket is mounted to the defender
	# Skip mount if docker is mounted from its default place (/var/run)
	if [[ -e "/var/run/systemd/private" ]] && [[ "${container_runtime_socket_folder}" != "/var/run" ]] && [[ "${container_runtime_socket_folder}" != "/run" ]]; then
		additional_mounts+=" -v /var/run/systemd/private:/var/run/systemd/private "
	fi

	# Add audit daemon log dir to monitor seccomp audits
	# Skip mount if docker seccomp security option is invalid or if audit log file doesn't exists
	if [[ -e "/var/log/audit/audit.log" ]] && [[ $(${docker} info | grep "seccomp") ]]; then
		additional_mounts+=" -v /var/log/audit:/var/log/audit "
	fi

	additional_env+=" -e DOCKER_CLIENT_ADDRESS=${DOCKER_SOCKET} "

	# Defender container capabilities
	# NET_ADMIN  - Required for process monitoring
	# NET_RAW    - Required for iptables (CNNF, runtime DNS, WAAS). See: https://bugzilla.redhat.com/show_bug.cgi?id=1895032
	# SYS_ADMIN  - Required for filesystem monitoring
	# SYS_PTRACE - Required for local audit monitoring and for reading some of /proc files
	# SYS_CHROOT - Required for changing mount namespace using setns
	# (see https://github.com/docker/docker/issues/6607 and https://github.com/docker/docker/issues/7276)
	# IPC_LOCK   - Required for defender to use perf events. This capability ignores limits set by mlock, which can cause mmap to fail
	# https://www.kernel.org/doc/html/latest/admin-guide/perf-security.html#unprivileged-users
	caps="--cap-add=NET_ADMIN \
			--cap-add=NET_RAW \
			--cap-add=SYS_ADMIN \
			--cap-add=SYS_PTRACE \
			--cap-add=SYS_CHROOT \
			--cap-add=IPC_LOCK "

	if [ "${component}" == "${defender_component_podman}" ]; then
		additional_env+=" -e DEFENDER_TYPE=podman "
	fi

	if is_podman; then
		if [ "${component}" != "${defender_component_podman}" ]; then
			additional_env+=" -e DEFENDER_TYPE=cri "
		fi
		# MKNOD - a capability to create special files using mknod(2), used by docker-less registry scanning
		# SETFCAP - a capability to set file capabilities, used by docker-less registry scanning
		caps+=" --cap-add=MKNOD \
				--cap-add=SETFCAP "
	fi

	additional_parameters=""
	if [[ -n ${selinux_enabled} ]]; then
		additional_parameters+=" --security-opt label${security_opt_operator}${SELINUX_LABEL} "
	fi

	print_debug "admin address ${console_address}"

	# Mount docker internal network namespace folder
	if [ -e "/var/run/docker/netns" ]; then
		additional_mounts+=" -v /var/run/docker/netns:/var/run/docker/netns:ro "
	fi

	if [[ $(version_higher_equal ${docker_version} 1.10) == 1 ]] || is_podman; then
		# Set seccomp unconfined profile for docker version higher or equals than 1.10
		# Seccomp default profile: https://docs.docker.com/engine/security/seccomp/
		# Docker default profile blocks setns, which is required for file system monitoring
		additional_parameters+=" --security-opt seccomp${security_opt_operator}unconfined "
	fi

	# Disabling app-armor (if enabled) is required for accessing /proc interfaces
	# https://github.com/docker/docker/issues/7276#issuecomment-68812214
	# Keep legacy condition in case apparmor does not appear in docker-info security options
	if [[ "${host_os_distribution}" == "Ubuntu" && $(apparmor_status 2>/dev/null | grep loaded) ]] || [[ $(${docker} info 2>/dev/null | grep apparmor) ]]; then
		additional_parameters+=" --security-opt apparmor${security_opt_operator}unconfined "
	fi

	# Set pids limit to 1000 so attackers won't be able to launch a fork bomb with a single command inside the container
	if [[ $(version_higher_equal ${docker_version} 1.12) == 1 ]] || is_podman; then
		additional_parameters+=" --pids-limit 1000 "
	fi

	container_name="twistlock_defender${DOCKER_TWISTLOCK_TAG}"

	docker_action=""
	if [[ "${SYSTEMD_ENABLED}" == "true" ]]; then
		print_debug "Systemd enabled. Creating container"
		docker_action="create"
		echo "DEFENDER=${container_name}" >${DATA_FOLDER}/scripts/defender.conf
	else
		docker_action="run"
		additional_parameters+=" -d --restart=always "
	fi

	if [[ ${READ_ONLY_FS} != "false" ]]; then
		# https://blog.docker.com/2015/02/docker-1-5-ipv6-support-read-only-containers-stats-named-dockerfiles-and-more/
		print_debug "Setting Defender container to readonly"
		additional_parameters+=" --read-only=true "
	else
		print_warning "Defender container file-system is not read-only"
	fi

	# --pid=host is needed so defender will be able to read the host's /proc directory
	# Remark: Defender is the only container that has a name combined from container name and TAG
	${docker} ${docker_action} \
		$(get_userns_flag) \
		--name=${container_name} \
		--net=host \
		--pid=host \
		${caps} \
		-e WS_ADDRESS=${WS_ADDRESS} \
		-e INSTALL_BUNDLE=${install_bundle} \
		-e HOSTNAME=${DEFENDER_CN} \
		-e LOG_PROD=true \
		-e REGISTRY_SCAN_ENABLED=${registry_enabled} \
		-e DATA_FOLDER=${DATA_FOLDER} \
		-e SYSTEMD_ENABLED=${SYSTEMD_ENABLED} \
		-e HOST_CUSTOM_COMPLIANCE_ENABLED=${HOST_CUSTOM_COMPLIANCE_ENABLED} \
		-e CLOUD_HOSTNAME_ENABLED=${CLOUD_HOSTNAME_ENABLED} \
		-e FIPS_ENABLED=${FIPS_ENABLED} \
		${additional_env} \
		${additional_mounts} \
		-v ${DATA_FOLDER}:/var/lib/twistlock \
		-v ${syslog_mount}:${syslog_mount} \
		${additional_parameters} \
		--cpu-shares 900 \
		-m 512m \
		${defender_image} >>${install_log} 2>&1

	exit_on_failure "$?" "Failed to run Twistlock Defender"

	local retries=1
	if [[ ${SYSTEMD_ENABLED} == "true" ]]; then
		print_debug "Starting Twistlock Defender using systemd"
		install_systemd_service ${defender_service}
		retries=3
	fi

	validate_container_up "twistlock_defender${DOCKER_TWISTLOCK_TAG}" ${retries}
	print_info "Twistlock Defender installed successfully"
}

install_server_defender() {
	local install_folder=${1}
	local console_address=${2}
	local install_bundle=${3}

	if [ -z "${install_folder}" ]; then
		exit_now "No install folder was provided"
	fi

	if [ -z "${console_address}" ]; then
		exit_now "No console address was provided"
	fi

	print_info "Installing Linux Server Defender to ${install_folder}"

	print_debug "Install folder: ${install_folder}"
	print_debug "Console addresss: ${console_address}"
	print_debug "WS port: ${COMMUNICATION_PORT}"

	set_server_defender_install_folder_permissions "${install_folder}"
	mkdir -p "${DATA_FOLDER}/data"
	mkdir -p "${DATA_FOLDER}/log"
	set_data_folder_permissions
	copy_local_certificates

	# Unpack archive
	tar zxf "${defender_bundle_linux_server}" -C "${install_folder}"
	exit_on_failure $? "Failed extracting ${defender_bundle_linux_server}"

	init_system=is_init_system
	install_path="${install_folder}/${server_defender_service}"
	if ${init_system}; then
		install_path="${install_folder}/${server_defender_upstart}"
	fi

	mv "${install_path}" ${DATA_FOLDER}/scripts/ >>${install_log} 2>&1
	exit_on_failure $? "Failed to move Twistlock Defender service file"

	# Create environment file
	cat <<-EOF >${DATA_FOLDER}/scripts/defender.conf
		WS_ADDRESS=wss://${console_address}:${COMMUNICATION_PORT}
		HOSTNAME=${DEFENDER_CN}
		LOG_PROD=true
		DATA_FOLDER=${DATA_FOLDER}
		INSTALL_FOLDER=${install_folder}
		INSTALL_BUNDLE=${install_bundle}
		HOST_CUSTOM_COMPLIANCE_ENABLED=${HOST_CUSTOM_COMPLIANCE_ENABLED}
		CLOUD_HOSTNAME_ENABLED=${CLOUD_HOSTNAME_ENABLED}
		DEFENDER_TYPE=serverLinux
		FIPS_ENABLED=${FIPS_ENABLED}
	EOF

	# Replace placeholders in service file (these cannot use the values from the env file)
	pushd "${DATA_FOLDER}/scripts" >/dev/null
	if ${init_system}; then
		fill_service_placeholders ${server_defender_upstart}
	else
		fill_service_placeholders ${server_defender_service}
	fi

	mv "${install_folder}/version.txt" "${DATA_FOLDER}/data"
	copy_db_ip_database

	if ${init_system}; then
		print_debug "Starting Twistlock Linux Server Defender using upstart"
		install_upstart_defender_server
	else
		print_debug "Starting Twistlock Linux Server Defender using systemd"
		install_systemd_service ${server_defender_service}
	fi

	popd >/dev/null

	print_info "Twistlock Linux Server Defender installed successfully"
}

# Uninstall server defender keeps data folder intact, to keep the defender logs
uninstall_server_defender() {
	local install_folder=$1

	if [ -z "${install_folder}" ]; then
		exit_now "No install folder was provided"
	fi

	print_info "Uninstalling Twistlock Linux Server Defender"
	if is_init_system; then
		uninstall_upstart_defender_server
	else
		uninstall_systemd_service ${server_defender_service}
	fi

	print_info "Removing Twistlock install folder: ${install_folder}"
	rm -rf ${install_folder}

	print_info "Defender for Linux Server uninstalled successfully"
}

version_higher_equal() {
	ver=$(echo -ne "$1\n$2" | sort -Vr | head -n1)
	if [ "$2" == "$1" ]; then
		echo 1
	elif [ "$1" == "${ver}" ]; then
		echo 1
	else
		echo -1
	fi
}

# Copy local certificate files (if exists)
copy_local_certificates() {
	cert_dir=${DATA_FOLDER}/certificates
	print_debug "Certificate dir: ${cert_dir}"
	mkdir -m 600 -p "${cert_dir}"

	local console_exists=""
	if command_exists ${docker}; then
		console_exists=$(${docker} ps -aq -f name=twistlock_console)
	fi

	# Copy all the certificate files including the old defender certs, e.g. defender-ca.pem.old
	for cert in ./*.pem*; do
		print_debug "Copying local certificate ${cert}"
		# If console is running on the same machine, use twistlock user permissions
		if [ ${console_exists} ]; then
			chown ${twistlock_user} ${cert} >>${install_log} 2>&1
			chmod u+rw ${cert} >>${install_log} 2>&1
		fi
		mv "${cert}" "${cert_dir}" >>${install_log} 2>&1
	done

	if [ -f "service-parameter" ]; then
		# If console is running on the same machine, use twistlock user permissions
		if [ ${console_exists} ]; then
			chown ${twistlock_user} service-parameter && chmod u+rw service-parameter
		fi
		mv service-parameter ${cert_dir}/service-parameter
	fi
}
# Sets twistlock data folder permissions
set_data_folder_permissions() {
	print_debug "Setting root only permissions to data folder: ${DATA_FOLDER}"
	mkdir -p "${DATA_FOLDER}"
	chmod o-rwx -R "${DATA_FOLDER}" >>${install_log} 2>&1
}

# Copy db-ip database
copy_db_ip_database() {
	local data_folder="${DATA_FOLDER}/data"
	print_debug "Copying db-ip database to data folder: ${data_folder}"
	mkdir -p "${data_folder}"
	mv "${install_folder}/dbip-country.mmdb" "${data_folder}"
}

# Sets server defender install folder permissions
set_server_defender_install_folder_permissions() {
	local install_folder=$1

	print_debug "Setting root only permissions to install folder: ${install_folder}"
	mkdir -p ${install_folder}
	chown -R ${twistlock_user} ${install_folder}
	chmod o-rwx -R ${install_folder}
}

# Sets twistlock data recovery folder permissions
set_data_recovery_folder_permissions() {
	print_debug "Setting user permissions to data recovery folder: ${DATA_RECOVERY_VOLUME}"
	mkdir -p ${DATA_RECOVERY_VOLUME}
	chown -R ${twistlock_user} ${DATA_RECOVERY_VOLUME} && chmod u+rwx -R ${DATA_RECOVERY_VOLUME}
}

command_exists() {
	command -v "$@" >/dev/null 2>&1
}

# gets the default userns flag (userns=host)
get_userns_flag() {
	if [[ $(is_userns_enabled) == "true" ]]; then
		echo " --userns=host "
	fi
	echo ""
}

# checks whether the user namespace feature is enabled
is_userns_enabled() {
	if [[ $(ps aux | grep -q "\-\-userns-remap") ]] || [[ $(${docker} info | grep "userns") ]]; then
		echo "true"
	else
		echo "false"
	fi
}

# checks whether the user namespace feature is enabled and prints a debug messages if needed
check_userns_enabled() {
	if [[ $(is_userns_enabled) == "true" && $(version_higher_equal ${docker_version} 1.11) == -1 ]]; then
		print_error "Running Twistlock with userns support requires docker version > 1.10"
	fi
}

copy_from_container() {
	local container=$1
	local src=$2
	local dst=$3

	if is_podman; then
		# Podman does not support 'cp' command. We will emulate it by:
		# 1. 'podman mount' the container. This will expose the mounts of the container to the host
		# 2. Copy from destination in the container mount to the host destination
		# 3. 'podman unmount'
		local mnt=$(${docker} mount ${container} 2>>${install_log})
		exit_on_failure $? "Failed to mount container ${container}"
		cp ${mnt}${src} ${dst} >>${install_log} 2>&1
		exit_on_failure $? "Failed to copy from container ${container}:${src} to host at ${dst}"
		${docker} unmount ${container} >>${install_log} 2>&1
		return
	fi

	${docker} cp ${container}:${src} ${dst} >>${install_log} 2>&1
	exit_on_failure $? "Failed to copy from container ${container}:${src} to host at ${dst}"
}

# Load images from tarcd
load_images() {
	component=$1
	print_info "Loading ${component} images"
	# Add selinux security options for generated containers
	if [[ -n ${selinux_enabled} ]]; then
		additional_run_parameters+=" --security-opt label${security_opt_operator}${SELINUX_LABEL} "
	fi

	if [[ ${component} == "" || ${component} == "${onebox_component}" || ${component} == "${console_component}" ]]; then

		console_local_images=${working_folder}"/${console_bundle}"
		# Skip if image is loaded
		if [[ "$(${docker} images | awk '{print $1":"$2}' | grep ${console_image} 2>/dev/null)" == "" ]]; then
			if [[ ! -e ${console_local_images} ]]; then
				exit_now "Failed to find ${console_image} offline image (${console_bundle})"
			fi

			${docker} load <${console_local_images}
		else
			print_info "Image ${console_image} already loaded"
		fi

		print_debug "Pulling Twistlock data from data container"
		remove_container "twistlock_data"
		# Copy required installation files to local folder
		${docker} create --name twistlock_data -ti ${additional_run_parameters} --entrypoint=/bin/sh "twistlock/private:console${DOCKER_TWISTLOCK_TAG}" >>${install_log} 2>&1
		exit_on_failure $? "Failed to start Twistlock data container"
		if [[ ${component} != "${console_component}" ]]; then
			copy_from_container "twistlock_data" "/images/${defender_bundle_linux_docker}" "${working_folder}/"
		fi
		copy_from_container "twistlock_data" "${static_data_folder}/twistlock-console.service" "${DATA_FOLDER}/scripts/"

		remove_container "twistlock_data"
	fi

	if [[ ${component} == "" || ${component} == "${onebox_component}" || ${component} == "${defender_component_docker}" || ${component} == "${defender_component_podman}" ]]; then
		# Skip if image is loaded
		if [[ "$(${docker} images | awk '{print $1":"$2}' | grep ${defender_image} 2>/dev/null)" == "" ]]; then

			defender_local_images=${working_folder}"/${defender_bundle_linux_docker}"
			if [[ ! -e ${defender_local_images} ]]; then
				exit_now "Failed to find ${defender_image} offline image (${defender_bundle_linux_docker})"
			fi

			${docker} load <${defender_local_images}
		else
			print_info "Image ${defender_image} already loaded"
		fi
		print_debug "Copying Twistlock Defender data to utils folder"
		mkdir -p ${DATA_FOLDER}/utils
		${docker} create --name twistlock_data -ti ${additional_run_parameters} --entrypoint=/bin/sh "twistlock/private:defender${DOCKER_TWISTLOCK_TAG}" >>${install_log} 2>&1
		exit_on_failure $? "Failed to create Twistlock data container"
		copy_from_container "twistlock_data" "${static_data_folder}/${defender_service}" "${DATA_FOLDER}/scripts/"
		remove_container "twistlock_data"

		# Replace placeholders in service file (these cannot use the values from the env file)
		fill_service_placeholders "${DATA_FOLDER}/scripts/${defender_service}"
	fi
}

# Checks the selinux mode of the docker daemon (Enforcing or not), and copies the policy
# If selinux_enabled is already provided as a parameter, then it is assumed that it's correct and uses it.
set_selinux_configurations() {
	# Check if selinux is enabled by inspecting docker daemon arguments and docker info
	# If podman is used, product is deployed on RedHat OS, so SELINUX is enabled
	if [[ $(ps aux | grep docker | grep "selinux-enabled") ]] || [[ $(${docker} info 2>/dev/null | grep "selinux") ]] || is_podman; then
		print_debug "Docker engine is running with SELinux enabled"
		selinux_enabled="true"
	else
		print_debug "Docker engine is running with SELinux disabled"
		return
	fi
}

upstart_docker_config="/etc/default/docker"
# Default docker configuration in RHEL
docker_sysconfig="/etc/sysconfig/docker"

exit_on_failure() {
	rc=$1
	message=$2
	if [[ ${rc} != 0 ]]; then
		# Also print the last line
		logs=$(tail --lines=2 ${install_log})
		print_error "${logs}"
		print_error "${message}"
		exit ${rc}
	fi
}

# stop the program and exit with the given message
exit_now() {
	local message=$1
	print_error "${message}"
	exit 1
}

convert_kb_to_gb() {
	let "kb_in_gb = 1024*1024"

	# rounding to the closest integer
	echo "$((($1 + (${kb_in_gb} / 2)) / (${kb_in_gb})))"
}

check_sys_requirements() {
	target=$1
	print_info "Performing system checks for ${target} mode.."
	int_re='^[0-9]+$'
	print_debug "Checking hardware requirements.."
	case ${target} in
	"${onebox_component}")
		# Minimal system requirements to run onebox, derived from registry scan requirements:
		let "min_storage_size_in_1kb_blocks = 18*1024*1024"
		let "min_ram_size_in_1kb_blocks     = 1024*1024 + 256*1024"
		let "min_cpu_count                  = 2"
		let "cpu_count = 1 +$(cat /proc/cpuinfo | grep "processor" | grep ":" | tail -1 | awk '{print $3}')"

		if [ "${min_cpu_count}" -gt "${cpu_count}" ]; then
			print_warning "The ${target} mode requires ${min_cpu_count} CPUs; The host has only ${cpu_count}"
		fi
		;;
	"${console_component}")
		# Minimal system requirements to run console:
		let "min_storage_size_in_1kb_blocks = 10*1024*1024"
		let "min_ram_size_in_1kb_blocks     = 1024*1024"
		;;
	"${defender_component_docker}" | "${defender_component_linux_server}")
		# Minimal system requirements to run defender:
		let "min_storage_size_in_1kb_blocks = 8*1024*1024"
		let "min_ram_size_in_1kb_blocks     = 256*1024"
		;;
	esac

	# Get the available free disk space of the installation folder's volume
	storage_capacity=$(df ${DATA_FOLDER} | tail -1 | awk '{print $4}')

	# Some older df version add a wrong line break, so in such cases we skip the test
	if ! [[ ${storage_capacity} =~ ${int_re} ]]; then
		print_warning "'df' returned an invalid output"
		print_debug "$(df ${DATA_FOLDER} | tail -1 | awk '{print "df output:", $0; print "capacity:", $4}')"
	fi

	if [[ ${storage_capacity} =~ ${int_re} ]] && [ "${storage_capacity}" -lt "${min_storage_size_in_1kb_blocks}" ]; then
		required_storage_gb=$(convert_kb_to_gb ${min_storage_size_in_1kb_blocks})
		actual_storage_gb=$(convert_kb_to_gb ${storage_capacity})
		deficit_gb=$((${required_storage_gb} - ${actual_storage_gb}))

		if [ "${deficit_gb}" -gt "0" ]; then
			print_warning "Low host storage space; Required: ${required_storage_gb}[GB], actual: ${actual_storage_gb}[GB]"
			let "low_storage_bound_console_gb = 20"
			if [[ "${target}" != "${defender_component_docker}" ]] && [[ "${target}" != "${defender_component_linux_server}" ]] && [[ "${hard_block}" != "false" ]] && [[ "${actual_storage_gb}" -le "${low_storage_bound_console_gb}" ]]; then
				exit_now "Blocked Twistlock ${target} installation; Low storage (less than ${low_storage_bound_console_gb}[GB]); To bypass rerun with -y"
			fi
		fi
	fi

	ram_capacity=$(cat /proc/meminfo | grep "MemTotal:" | awk '{print $2}')
	if [ "${ram_capacity}" -lt "${min_ram_size_in_1kb_blocks}" ]; then
		required_ram_gb=$(convert_kb_to_gb ${min_ram_size_in_1kb_blocks})
		actual_ram_gb=$(convert_kb_to_gb ${ram_capacity})
		deficit_gb=$((${required_ram_gb} - ${actual_ram_gb}))

		if [ "${deficit_gb}" -gt "0" ]; then
			print_warning "Low host RAM capacity; Required: ${required_ram_gb}[GB], actual: ${actual_ram_gb}[GB]"
		fi
	fi
}

merge_configuration_files() {
	local old_configuration="${DATA_FOLDER}/scripts/twistlock.cfg"
	if [ ! -e "${old_configuration}" ]; then
		exit_now "Failed to find old configuration file ${old_configuration}"
	fi

	print_info "Using previously user-defined settings in ${old_configuration}"
	while IFS='' read -r line || [[ -n "${line}" ]]; do
		if [[ ${line} == "" || "${line:0:1}" == "#" ]]; then
			continue
		fi
		tokens=$(echo ${line} | tr "=" "\n")
		var=$(echo ${tokens} | awk '{print $1}')

		# Skip Twistlock version
		if [ "${var}" == "DOCKER_TWISTLOCK_TAG" ]; then
			continue
		fi

		sed -i "s~^${var}=.*$~${line}~g" ${config_file}

		# The clustered DB config key is not set by default in the configuration file. In case it is missing after merging,
		# append it to the config so the setting persists on upgrade (the feature cannot be disabled).
		if [[ "${var}" == "CLUSTERED_DB_ENABLED" && ! $(grep ${var} ${config_file}) ]]; then
			echo ${line} >>${config_file}
		fi
	done <"${DATA_FOLDER}/scripts/twistlock.cfg"
	source ${config_file}
}

echo "$(tput_silent setaf 3)
  _____          _     _   _            _
 |_   _|_      _(_)___| |_| | ___   ___| | __
   | | \ \ /\ / / / __| __| |/ _ \ / __| |/ /
   | |  \ V  V /| \__ \ |_| | (_) | (__|   <
   |_|   \_/\_/ |_|___/\__|_|\___/ \___|_|\_\\

$(tput_silent sgr0)"

show_help() {
	echo "
Usage: sudo twistlock.sh [OPTIONS] [COMPONENT]

Twistlock.sh installs and configures Twistlock in your environment.

OPTIONS:
  -f                            Drops existing database
  -h                            Shows help
  -j                            Merge previous Twistlock configuration file (default to override)
  -s                            Agrees to the EULA
  -u                            Uninstalls Twistlock (removes all Twistlock containers and images)
  -y                            Bypasses installation blocks even when system requirements are not met
  -z                            Outputs debug logs


COMPONENT:
  onebox                        Installs all Twistlock components (Console and Defender) on a single host

"
}

args="$@"
# Internals:
# -a    [console_address]       Console server address for communication with Twistlock defender
# -n                            Install all containers on separate networks (default false)
# -p                            Generate server certs files locally
# -t                            Override twistlock.cfg image tag
# -x                            Path to docker client
# -o                            Docker client options (e.g., -H). Must be provided after client option
OPTIND=1
unset name
optspec="h?a:ksdb:fjo:ruzw:t:x:i:e:b:y-:"
while getopts "${optspec}" opt; do
	case "${opt}" in
	-)
		case "${OPTARG}" in
		install-folder)
			# Use indirect parameter expansion to get the value of a positional parameter, e.g.,
			# if OPTIND=3 then ${!OPTIND} evaluates to the value of $3
			install_folder="${!OPTIND}"
			OPTIND=$((${OPTIND} + 1))
			;;
		install-data-folder)
			install_data_folder="${!OPTIND}"
			OPTIND=$((${OPTIND} + 1))
			;;
		ws-port)
			ws_port="${!OPTIND}"
			OPTIND=$((${OPTIND} + 1))
			;;
		podman-socket)
			additional_env+=" -e PODMAN_CLIENT_ADDRESS=${!OPTIND}"
			OPTIND=$((${OPTIND} + 1))
			;;
		env)
			additional_env+=" -e ${!OPTIND} "
			OPTIND=$((${OPTIND} + 1))
			;;
		*)
			if [ "${OPTERR}" = 1 ] && [ "${optspec:0:1}" != ":" ]; then
				echo "Unknown option --${OPTARG}" >&2
			fi
			;;
		esac
		;;
	h)
		show_help
		exit
		;;
	e)
		custom=${OPTARG}
		;;
	a)
		console_address=${OPTARG}
		;;
	s)
		skip_eula=true
		;;
	f)
		drop="true"
		;;
	r)
		registry_enabled="true"
		;;
	t)
		tag=${OPTARG}
		;;
	i)
		host_os_distribution=${OPTARG}
		;;
	u)
		uninstall="true"
		;;
	z)
		debug_log_level="true"
		;;
	x)
		docker=${OPTARG}
		;;
	o)
		docker+="${OPTARG//\"/}"
		# Additional client options (remove " to enable specifying multiple parameters)
		;;
	y)
		hard_block="false"
		# When hard_block is set to 'false' we won't stop installation upon low storage
		;;
	j)
		merge_previous_cfg="true"
		;;
	b) install_bundle=${OPTARG} ;;
	esac
done

shift $((OPTIND - 1))
[ "$1" = "--" ] && shift

# Merge with previous config file if exists
if [ "${merge_previous_cfg}" == "true" ]; then
	merge_configuration_files
fi

# Run console as root or as service user
# Remark: Use configuration parameters only after previous configuration merge
if [[ ${RUN_CONSOLE_AS_ROOT} == "true" ]]; then
	twistlock_user=root
else
	twistlock_user=2674
fi

# Override config values if given as arguments
if [[ -n "${install_data_folder}" ]]; then
	DATA_FOLDER=${install_data_folder}
fi

if [[ -n "${ws_port}" ]]; then
	COMMUNICATION_PORT=${ws_port}
fi

if [ -z ${DEFENDER_CN} ]; then
	DEFENDER_CN=$(hostname --fqdn 2>/dev/null)
	if [[ $? == 1 ]]; then
		DEFENDER_CN=$(hostname)
	fi
fi

component=${BASH_ARGV}
print_debug "${component} - ${args} (config: ${config_file})"
print_debug "Using Docker command: ${docker}"

# Check if last parameter is argument (-u)
if [[ ${component} == -* ]] && [[ ${uninstall} == "true" ]]; then
	argument="true"
	component=""
fi

# Exit if no component is specified
if (
	[[ -z "${custom}" ]] &&
		[ "${component}" != "${onebox_component}" ] &&
		[ "${component}" != "${defender_component_docker}" ] &&
		[ "${component}" != "${defender_component_podman}" ] &&
		[ "${component}" != "${defender_component_linux_server}" ] &&
		[ "${component}" != "${defender_component_tas}" ] &&
		[ "${component}" != "${console_component}" ] &&
		[[ "${argument}" != "true" ]]
); then
	show_help
	exit
fi

# Create Twistlock scripts folder if not exists
mkdir -p ${DATA_FOLDER}/scripts

if [[ "${component}" == "${defender_component_linux_server}" ]]; then
	if [[ -z "${install_folder}" ]]; then
		install_folder="/opt/twistlock"
	fi

	if [[ "${uninstall}" == "true" ]]; then
		uninstall_server_defender "${install_folder}"
		exit
	fi

	print_debug "Checking host system requirements"
	check_sys_requirements "${component}"

	copy_scripts

	install_server_defender "${install_folder}" "${console_address}" "${install_bundle}"
	exit
fi

if [[ "${component}" == "${defender_component_tas}" ]]; then
	DATA_FOLDER=${TAS_DATA_DIR}
	if [ -d ${DATA_FOLDER} ]; then
		mkdir ${DATA_FOLDER}
	fi
	install_folder=${TAS_BINARY_DIR}
	twistlock_user=root

	# Unpack archive
	print_debug "Extracting the defender to ${install_folder}"
	tar zxf "${defender_bundle_linux_server}" -C "${install_folder}"
	exit_on_failure $? "Failed extracting ${defender_bundle_linux_server}"

	set_server_defender_install_folder_permissions "${install_folder}"
	# Since TAS defender runs as a service, static data folder must be created
	mkdir -p "${DATA_FOLDER}/data"
	set_data_folder_permissions
	copy_local_certificates
	copy_db_ip_database

	export WS_ADDRESS=wss://${console_address}:${COMMUNICATION_PORT}
	export LOG_PROD=true
	export DATA_FOLDER=${DATA_FOLDER}
	export INSTALL_BUNDLE=${install_bundle}
	export FORCE_UNMOUNT_SHUTDOWN=${FORCE_UNMOUNT_SHUTDOWN}

	if [[ ${working_folder} == *.twistlock ]]; then
		# Cleanup working folder data because the defender execution will be blocked
		rm "${working_folder}"/*
	fi

	print_info "Starting TAS defender"
	${TAS_BINARY_DIR}/defender 2>&1 >>${TAS_LOG_DIR}/defender.stdout.log

	exit
fi

set_docker_command ${component}
set_docker_version

min_version=1.7
if is_podman; then
	min_version=1.2
fi

if [[ $(version_higher_equal ${docker_version} ${min_version}) != 1 ]]; then
	if is_podman; then
		docker="podman"
	else
		docker="docker"
	fi
	print_error "Twistlock requires ${docker} version higher than ${min_version}"
	exit 1
fi
# Legacy security options operator (<= 1.10)
if [[ $(version_higher_equal ${docker_version} 1.11) == -1 ]] && is_docker; then
	security_opt_operator=":"
fi

if [ "${tag}" ]; then
	print_debug "Using Twistlock image tag - ${tag}"
	DOCKER_TWISTLOCK_TAG=${tag}
fi

defender_image="twistlock/private:defender${DOCKER_TWISTLOCK_TAG}"
console_image="twistlock/private:console${DOCKER_TWISTLOCK_TAG}"

if [ "${uninstall}" == "true" ]; then
	uninstall_twistlock
	exit
fi

if [ "${skip_eula}" != true ]; then
	approve_disclaimer
fi

set_data_folder_permissions

print_debug "Checking host system requirements"
check_sys_requirements ${component}

set_selinux_configurations

load_images ${component}

check_userns_enabled

copy_scripts

if [ "${component}" == "${console_component}" ]; then
	if [[ $(${docker} ps | grep 'defender') ]]; then
		exit_now "In a onebox configuration, Defender and Console must be upgraded together"
	fi
	install_console "${console_address}" "${drop}"
fi

if [ "${component}" == "${onebox_component}" ]; then
	install_console "${console_address}" "${drop}"
	# When the console is installed and launched it creates all the certificates required for the console API and communication
	# with the defenders. In onebox configuration the defender is installed without the certificates bundle. It uses the
	# certificates which are created by the console and located on the same machine, therefore wait for all certificates to
	# be created by the console before installing the defender
	for cert_name in defender-ca.pem defender-client-cert.pem defender-client-key.pem ca.pem server-cert.pem server-key.pem; do
		cert=${DATA_FOLDER}/certificates/${cert_name}
		while [ ! -f "$cert" ]; do
			print_info "The certificate file ${cert} does not exist. Sleeping 1 sec"
			sleep 1
		done
	done

	# Always install registry scan capabilities in onebox
	# 127.0.0.1 is used instead of hostname since there are problems reaching localhost hostname inside defender
	install_defender "127.0.0.1" "true"

	print_info "Installation completed successfully"
fi

if [ "${component}" == "${defender_component_docker}" ] ||
	[ ${component} == "${defender_component_podman}" ]; then
	if [ -z "${console_address}" ]; then
		print_error "Console address parameter must be provided"
		exit
	fi

	install_defender "${console_address}" "${registry_enabled}" "${install_bundle}" "${component}"
	exit
fi

shift $((OPTIND - 1))
[ "$1" = "--" ] && shift

exit 0
